<?php namespace App\Controllers\Api\Ipn;

use App\Controllers\BaseController;
use App\Libraries\Uid;
use App\Models\AppsModel;
use App\Libraries\Settings;
use App\Models\DepositMethodsModel;
use App\Models\PlansModel;
use App\Models\SubscribesModel;
use App\Models\TransactionsModel;
use App\Models\PlansExtraModel;
use App\Models\SubscribesExtraModel;
use App\Models\TransactionsExtraModel;
use CodeIgniter\HTTP\ResponseInterface;
use CodeIgniter\Database\Exceptions\DatabaseException;

class PayPal extends BaseController
{
    protected $paypalUrl = 'https://api-m.sandbox.paypal.com/v1/notifications/verify-webhook-signature';
    protected $method;

    public function __construct()
    {
        $methods = new DepositMethodsModel();
        $this->method = $methods->where([
            "name" => "PayPal",
            "status" => 1
        ])->first();

        if (!$this->method) {
            throw new DatabaseException(lang("Message.message_84"));
        }
    }

    public function index(): ResponseInterface
    {
        try {
            $payload = @file_get_contents('php://input');
            $this->verifyWebhookSignature($payload);

            $event = json_decode($payload, true);

            // Xác định loại sự kiện và lấy custom_id
            $customId = null;
            if (isset($event['resource'])) {
                $resource = $event['resource'];
                $customId = $resource['custom_id'] ?? null;
                $eventType = $event['event_type'] ?? null;
            } elseif (isset($event['id']) && isset($event['purchase_units'])) {
                $resource = $event['purchase_units'][0]['payments']['captures'][0];
                $customId = $resource['custom_id'] ?? null;
                $eventType = $resource['status'] ?? null;
            } else {
                throw new \Exception("Invalid event data");
            }

            if ($customId === null) {
                throw new \Exception("Custom ID not found in event data");
            }

            // Phân biệt giữa thanh toán plan và plan_extra
            if (strpos($customId, 'EXTRA_') === 0) {
                $this->processEventExtra($resource, $eventType);
            } else {
                $this->processEvent($resource, $eventType);
            }

            return $this->respond(["status" => "ok"], 200);
        } catch (\Exception $e) {
            return $this->respond(["message" => $e->getMessage()], 400);
        }
    }

    private function verifyWebhookSignature($payload)
    {
        $client = service('curlrequest');
        $clientId = $this->method["api_value_1"];
        $clientSecret = $this->method["api_value_2"];
        $auth = base64_encode("$clientId:$clientSecret");

        $response = $client->post($this->paypalUrl, [
            'headers' => [
                'Authorization' => "Basic $auth",
                'Content-Type' => 'application/json'
            ],
            'json' => [
                'transmission_id' => $_SERVER['HTTP_PAYPAL_TRANSMISSION_ID'],
                'transmission_time' => $_SERVER['HTTP_PAYPAL_TRANSMISSION_TIME'],
                'cert_url' => $_SERVER['HTTP_PAYPAL_CERT_URL'],
                'auth_algo' => $_SERVER['HTTP_PAYPAL_AUTH_ALGO'],
                'transmission_sig' => $_SERVER['HTTP_PAYPAL_TRANSMISSION_SIG'],
                'webhook_id' => $this->method["api_value_3"],
                'webhook_event' => json_decode($payload, true)
            ]
        ]);

        $verification = json_decode($response->getBody(), true);

        if ($verification['verification_status'] !== 'SUCCESS') {
            throw new DatabaseException("Invalid webhook signature");
        }
    }

    private function processEvent($resource, $eventType)
    {
        $db = \Config\Database::connect();
        $db->transStart();

        try {
            switch ($eventType) {
                case 'PAYMENT.CAPTURE.COMPLETED':
                case 'COMPLETED':
                case 'CHECKOUT.ORDER.APPROVED':
                	$this->handlePaymentCaptureCompleted($resource);
                    break;
                default:
                    // Unhandled event type
                    log_message('info', 'Unhandled PayPal event type: ' . $eventType);
            }

            $db->transComplete();
        } catch (\Exception $e) {
            $db->transRollback();
            throw $e;
        }
    }
  
    private function processEventExtra($resource, $eventType)
    {
        $db = \Config\Database::connect();
        $db->transStart();

        try {
            switch ($eventType) {
                case 'PAYMENT.CAPTURE.COMPLETED':
                case 'COMPLETED':
                case 'CHECKOUT.ORDER.APPROVED':
                    $this->handlePaymentExtraCaptureCompleted($resource);
                    break;
                default:
                    log_message('info', 'Unhandled PayPal extra event type: ' . $eventType);
            }

            $db->transComplete();
        } catch (\Exception $e) {
            $db->transRollback();
            throw $e;
        }
    }

    private function handlePaymentCaptureCompleted($resource)
    {
        $subscribes = new SubscribesModel();
        $plans = new PlansModel();
        $transactions = new TransactionsModel();
        $uid = new Uid();

        $customID = $resource['custom_id'];
        $customValues = explode("_", $customID);

        $plan_id = $customValues[0];
        $user_id = $customValues[1];
        $app_id  = $customValues[2];

        // Lấy thông tin từ bảng plans bao gồm build_count
        $plan = $plans->where("id", $plan_id)->select("count,price,build_count")->first();
        if (!$plan) {
            throw new \Exception("Plan not found");
        }

        $amount_total = $plan["price"];
        $currentDate = date('Y-m-d');
        $newDate = date('Y-m-d', strtotime('+' . $plan['count'] . ' month', strtotime($currentDate)));

        // Thêm đăng ký mới vào bảng subscribes với remaining_count là giá trị của build_count
        $subscribes->insert([
            "subscribe_external_id" => $resource['id'],
            "customer_external_id"  => $resource['id'],
            "plan_id"               => $plan_id,
            "user_id"               => $user_id,
            "expires_at"            => strtotime($newDate),
            "app_id"                => $app_id,
            "price"                 => $amount_total,
            "uid"                   => $uid->create(),
            "is_active"             => 1,
          	"remaining_count"		=> $plan["build_count"],
            "method_id"             => $this->method['id']
        ]);

        // Thêm giao dịch vào bảng transactions
        $transactions->insert([
            'uid' => $uid->create(),
            'amount' => $amount_total,
            'status' => 1,  // Thành công
            'created_at' => time(),
            'updated_at' => time(),
            'method_id' => $this->method['id'],
            'subscribe_external_id' => $resource['id'],
            'external_uid' => $resource['id']
        ]);
    }
  
    private function handlePaymentExtraCaptureCompleted($resource)
    {
        $projects = new AppsModel();
        $uid = new Uid();

        $plansExtra = new PlansExtraModel();
        $transactions_extra = new TransactionsExtraModel();
        $subscribes_extra = new SubscribesExtraModel();

        $customID = $resource['custom_id'];
      	// Loại bỏ tiền tố 'EXTRA_'
        if (strpos($customID, 'EXTRA_') === 0) {
            $customID = substr($customID, strlen('EXTRA_'));
        } else {
            throw new \Exception("Invalid custom ID format");
        }
        $customValues = explode("_", $customID);

        // Loại bỏ tiền tố 'EXTRA_'
        $plan_extra_id = $customValues[0];
        $user_id = $customValues[1];
        $app_id  = $customValues[2];

        $plan_extra = $plansExtra->where("id", $plan_extra_id)->select("build_count_extra,price")->first();
      	if (!$plan_extra) {
            throw new \Exception("Plan extra not found");
        }
        $amount_extra_total = $plan_extra["price"];
        $currentDate = date('Y-m-d');

        // Thêm đăng ký mới vào bảng `subscribes_extra`
        $subscribes_extra->insert([
            "subscribe_extra_external_id" => $resource['id'],
            "customer_extra_external_id"  => $resource['id'],
            "plan_extra_id"               => $plan_extra_id,
            "user_id"                     => $user_id,
            "app_id"                      => $app_id,
            "price"                       => $amount_extra_total,
            "uid"                         => $uid->create(),
            "is_active"                   => 1,
            "method_id"                   => $this->method['id'],
          	"build_count_extra"           => $plan_extra['build_count_extra']
        ]);

        // Thêm giao dịch vào bảng `transactions_extra`
        $transactions_extra->insert([
            'uid' => $uid->create(),
            'amount' => $amount_extra_total,
            'status' => 1,  // Thành công
            'created_at' => time(),
            'updated_at' => time(),
            'method_id' => $this->method['id'],
            'subscribe_extra_external_id' => $resource['id'],
            'external_extra_uid' => $resource['id']
        ]);

        // Cập nhật remaining_count trong bảng subscribes
        $subscribes = new SubscribesModel();
        $subscribes->set('remaining_count', 'remaining_count + ' . $plan_extra["build_count_extra"], false)
                   ->where("user_id", $user_id)
                   ->where("app_id", $app_id)
                   ->update();
    }

    public function capture(): ResponseInterface
    {
        try {
            $clientId = $this->method["api_value_1"];
            $clientSecret = $this->method["api_value_2"];
            $auth = base64_encode("$clientId:$clientSecret");
            $orderId = $this->request->getGet('token');
            $paypalUrl = 'https://api-m.sandbox.paypal.com/v2/checkout/orders';
            $client = service('curlrequest');
            $response = $client->post("$paypalUrl/{$orderId}/capture", [
                'headers' => [
                    'Authorization' => "Basic $auth",
                    'Content-Type' => 'application/json',
                ]
            ]);

            if (!in_array($response->getStatusCode(), [200, 201])) {
                throw new \Exception("Failed to capture PayPal order. Status Code: " . $response->getStatusCode());
            }

            // Không gọi đến handlePaymentCaptureCompleted()
            // Chỉ chuyển hướng khách hàng

        } catch (\Exception $e) {
            log_message('error', 'PayPal Capture Error: ' . $e->getMessage());
            return $this->respond(["message" => $e->getMessage()], 400);
        }
        $settings = new Settings();
        $frontUrl = $settings->get_config("site_url");
        return redirect()->to($frontUrl.'private/profile/subscribe');
    }

    public function captureExtra(): ResponseInterface
    {
        try {
            $clientId = $this->method["api_value_1"];
            $clientSecret = $this->method["api_value_2"];
            $auth = base64_encode("$clientId:$clientSecret");
            $orderId = $this->request->getGet('token');
            $paypalUrl = 'https://api-m.sandbox.paypal.com/v2/checkout/orders';
            $client = service('curlrequest');
            $response = $client->post("$paypalUrl/{$orderId}/capture", [
                'headers' => [
                    'Authorization' => "Basic $auth",
                    'Content-Type' => 'application/json',
                ]
            ]);

            if (!in_array($response->getStatusCode(), [200, 201])) {
                throw new \Exception("Failed to capture PayPal extra order. Status Code: " . $response->getStatusCode());
            }

            // Không gọi đến handlePaymentExtraCaptureCompleted()
            // Chỉ chuyển hướng khách hàng

        } catch (\Exception $e) {
            log_message('error', 'PayPal Capture Extra Error: ' . $e->getMessage());
            return $this->respond(["message" => $e->getMessage()], 400);
        }
        $settings = new Settings();
        $frontUrl = $settings->get_config("site_url");
        return redirect()->to($frontUrl.'private/profile/subscribe_extra');
    }
}
